%% Default values for variables describing the simulated 3 row, fixed rack 
% system with modules in portrait or landscape.  All dimensions in mm.

if ~exist('tilt', 'var')
    tilt = 30;  % default value of array tilt from horizontal
end

if ~exist('elev', 'var')
    elev = 500; % default height of center of array in meters
end

if ~exist('row_spacing', 'var')
    row_spacing = 2000;
end

if ~exist('module_orientation', 'var')
    module_orientation = 'portrait';
end

if ~exist('module_spacing', 'var')
    module_spacing = 50; % distance between edges of adjacent modules
end

if ~exist('SurfaceAzimuth', 'var')
    SurfaceAzimuth = 180; % azimuth of array's normal projected to the ground
end


%% Translate from in-plane coordinates to array coordinates.

az = SurfaceAzimuth - 90; % azimuth in ground coordinate system of vector from left to right.

% Rotation angle from x-axis in ground coordinates to module width dimension.
rot = 90-az;
vw = [cosd(rot) sind(rot) 0];  % unit vector along module width direction in ground coordinates.

% Rotation matrix to be used to translate from in-plane coordinates to ground coordinates.
% Rodriguez formula from wikipedia.org/wiki/Rotation_matrix
R90z = [0 -1 0; 1 0 0; 0 0 1];

C = cosd(-tilt);
S = sind(-tilt);

Rvw = [0 -vw(3) vw(2); vw(3) 0 -vw(1); -vw(2) vw(1) 0]*S + (eye(3) - vw'*vw)*C + vw'*vw;

vl = vw*R90z'*Rvw; % unit vector along module length dimension in ground coordinates.

vlg = [vl(1:2) 0] / norm([vl(1:2) 0]);  % unit vector along ground in vl direction

%% Setup the base module with dimensions in meters.

objects = [];

% Define vectors from module's lower left corner in width (left-to-right) 
% and length (bottom to top) dimensions.
if strcmp(module_orientation, 'portrait')
    vModuleWidth = module_width*vw;  % vector from side to side along width (lateral) direction
    vModuleLength = module_height*vl; % vector from bottom to top along length (vertical) direction
    vModuleHeight = [0 0 elev];  % elev is height of lower left corner.
elseif strcmp(module_orientation, 'landscape')
    vModuleWidth = module_height*vw;  % vector from side to side along width (lateral) direction
    vModuleLength = module_width*vl; % vector from bottom to top along length (vertical) direction
    vModuleHeight = [0 0 elev];  % elev is height of lower left corner.
else
    ['Error in module_orientation = ' module_orientation]
    exit
end

%% Create each row of module objects

ctr = 0;

for row=1:num_rows % number of rows
    row_corner = (row - 1) * row_spacing*vlg;
    for pos=1:num_modules_per_row % number of modules in each row
        ctr = ctr + 1;
        objects(ctr).LL = row_corner + (pos - 1)*(vModuleWidth + module_spacing*vw) + vModuleHeight;
        objects(ctr).LR = objects(ctr).LL + vModuleWidth;
        objects(ctr).UL = objects(ctr).LL + vModuleLength;
        objects(ctr).UR = objects(ctr).LL + vModuleWidth + vModuleLength;
        objects(ctr).Color = [0 0 0.5];
    end
end

% all objects are coplanar. set up ground-facing normal

n1 = -[sind(tilt)*cosd(-az) sind(tilt)*sind(-az) cosd(tilt)];

for i=1:length(objects)
    objects(i).normal = n1;
end

% list of objects which cast shadows before/after solar noon
before_noon = [1:length(objects)];
after_noon = before_noon;


if make_figure
    % draw objects
    figure
    hold all
    ax = gca;
    for ctr=1:length(objects)
        draw_object(ax, objects(ctr), false)
    end
    draw_northarrow(ax);
    view(3)
    axis equal
    grid on
end

clearvars row_corner R90z rot Rvw row pos C S ctr n1
