%% Simulating rear-surface irradiance for bifacial modules in a 3 row, 
% fixed tilt system.
%
% This example simulates rear-surface irradiance for each cell of several 
% bifacial modules in a notional fixed tilt system comprising three rows 
% of eight modules each. Irradiance is simulated for a clear sky day in 
% Albuquerque, NM.
% 
% Rear-surface irradiance is simulated for each cell of the bifacial modules
% and for each of the three rear-facing reference cells. Rear-surface 
% irradiance comprises ground-reflected irradiance, calculated using view 
% factors from the ground to each cell, and sky diffuse irradiance 
% calculating using view factors from each cell to the visible sky. Shadows
% on the ground from modules and the rack's structure reduce the ground-
% reflected irradiance, and the modules and rack structure block part of 
% the sky diffuse irradiance. 
%
% The coordinate system's origin is at the earth surface directly below the
% the lower-left corner of the front row, left module. The x-axis points 
% from left to right along the rack's plane and parallel to the earth's 
% surface, the y-axis lies 90 degrees clockwise from the x-axis and 
% parallel to the earth's surface, and the z-axis points upwards. All 
% dimensions are in mm. 

clearvars
make_figure = true; % controls whether diagrams are made

% Add paths to local code and data
folder = fileparts(which('Example_simulate_fixed_tilt_system.m'));
addpath([folder '\InPolygon'])

%% Set up the array. We want to able to vary row spacing, height and tilt.

%%
% Array parameters.
tilt = 30; % tilt angle of array in degrees from horizontal.

%%
% Set the array azimuth, i.e., azimuth of the array normal when 
% projected to the ground.
SurfaceAzimuth = 180;

%%
% Determine the elevation (z coordinate) of the array's lower left 
% corner. The coordinate system's origin lies directly below the 
% array's lower left corner. |elev| is the height in mm of the array 
% over the grid origin. Data in |elev_setting| are the height of the 
% array center in meters.
elev = 500; 

%%
% Row spacing is the distance between lower left corners of modules in
% adjacent rows.
row_spacing = 3000;

%%
% Module spacing is the distance between edges of adjacent modules.
module_spacing = 50;

%%
% Set up |objects|.
% Structure array |objects| describes the objects which cast shadows. In
% this example, |objects| contains 24 entries, one for each module. Other 
% objects, e.g., the array racking, are not represented.
%
% Each element in |objects| has the following fields:
%
% * |LL|, |LR|, |UL|, |UR| : coordinates (x, y, z) of each corner (mm)
% * |Color| : (r, g, b) value to use when drawing figures
% * |normal| : (x, y, z) unit normal vector of the object in the
%        direction of the ground (mm)

module_width = 1041;  % module dimension (mm) left to right in portrait
module_height = 1651;  % module dimension (mm) bottom to top in portrait

num_cells_width = 6; % number of cells along module's width dimension
num_cells_length = 10; % number of cells along module's length dimension

cell_width = 6.25*25.4; % cell dimension (mm) left to right
cell_length = cell_width; % cell dimension (mm) bottom to top
cell_area = cell_width*cell_length;

cell_offset_from_left_edge = 44; % distance from module's left edge to edge of cell
cell_offset_from_bottom = 32; % distance from module's bottom edge to edge of cell

num_rows = 3;
num_modules_per_row = 8;

module_orientation = 'landscape';  % or 'landscape'
setup_fixed_tilt_array_modules;

%%
% Set up |cells|.
% Structure array |cells| defines the areas at which rear-surface 
% irradiance is simulated. In this example, |cells| contains 123
% surfaces: 60 cells for each of two bifacial modules, and 3
% rear-facing reference cells.
%
% Each element in |cells| has the following fields:
%
% * |name| : a string identifying the cell
% * |x|, |y|, |z| : coordinates of the cell's center (mm)
% * |normal| : (x, y, z) unit normal vector (mm) of the cell in the
%        direction of the ground
% * |width| : left-right dimension of the cell (mm)
% * |length| : bottom-top dimension of the cell (mm)
% * |area| : width x length (mm^2)
% * |vw| : unit vector from left to right along the width dimension
% * |vl| : unit vector from bottom to top along the length dimension
%

% Select modules where rear-surface irradiance will be simulated.  Format
% is [ [row, pos], ...] where row=1 is the front row, pos=1 is the
% left-most module of the row.
which_modules = [[1,1]; [2,4]];  % needs to be N x 2 array
setup_fixed_tilt_array_cells;


%% Simulate irradiance using Ineichen clear-sky model.

dt = datenum('2016-06-16 06:00:00'):1/(24*12):datenum('2016-06-16 18:00:00');
dv = datevec(dt);
hrofday = dv(:,4);

% Set ground albedo.
mean_albedo = 0.26; % 

% Set up |Location| structure for PVLib.
PSELLocation = pvl_makelocationstruct(35.05, -106.54, 1660);

% Calculate sun position at each time in preparation for projecting 
% objects to shadows on the ground.
PSELTime = pvl_maketimestruct(dt, -7);
[SunAzimuth, ~, SunElevation, SolarTime] = pvl_ephemeris(PSELTime, PSELLocation);

% Calcuate GHI, DNI and DHI using the Ineichen clear-sky model.
[GHI, DNI, DHI] = pvl_clearsky_ineichen(PSELTime, PSELLocation);

if make_figure
    figure
    hold all
    plot(dt, GHI)
    plot(dt, DNI)
    plot(dt, DHI)
    xlabel('Time of day')
    ylabel('Irradiance (W/m^2)')
    datetick('x','HH:MM','KeepTicks')
    legend('GHI','DNI','DHI','Location','best')
end

% Simulate irradiance for front and rear surfaces.
% Calculate angle-of-incidence AOI and beam irradiance on the plane-of-array.

AOI = pvl_getaoi(tilt, SurfaceAzimuth, 90-SunElevation, SunAzimuth);
BeamIrradiance = DNI .* cosd(AOI);
BeamIrradiance(cosd(AOI)<0) = 0; % cosd(AOI)<0 when sun is behind the plane-of-array

% Select model for sky diffuse irradiance.  Choices are |'isotropic'|,
% |'perez'| and |'haydavies'|.
SkyDiffuseModel = 'isotropic';

%%
% Calculate sky diffuse irradiance for front and rear surfaces.
HExtra = pvl_extraradiation(pvl_date2doy(PSELTime.year, PSELTime.month, PSELTime.day));
AMrelative = pvl_relativeairmass(90-SunElevation);
AMabsolute = pvl_absoluteairmass(AMrelative, pvl_alt2pres(PSELLocation.altitude));

if strcmp(SkyDiffuseModel, 'perez')
    FrontSkyDiffuse = pvl_perez(tilt, SurfaceAzimuth, DHI, ...
                                DNI, HExtra, 90-SunElevation, ...
                                SunAzimuth, AMabsolute);
    RearSkyDiffuse = DHI.*(1-cosd(tilt))/2;  % isotropic - need to replace with Perez, no horizon shading

elseif strcmp(SkyDiffuseModel, 'haydavies')
    FrontSkyDiffuse = pvl_haydavies1980(tilt, SurfaceAzimuth, DHI, ...
                                        DNI, HExtra, ...
                                        90-SunElevation, SunAzimuth);
    % Include only non-circumsolar component
    A = DNI./HExtra;
    RearSkyDiffuse = (1-A).*DHI.*(1-cosd(tilt))/2; % no horizon shading

elseif strcmp(SkyDiffuseModel, 'isotropic')                                    
    FrontSkyDiffuse = pvl_isotropicsky(tilt, DHI);
    RearSkyDiffuse = DHI.*(1-cosd(tilt))/2;  % isotropic, no horizon shading

else
    'Unrecognized or unspecified sky diffuse model'
    FrontSkyDiffuse = NaN;
    RearSkyDiffuse = NaN;
end

%%
% Calculate ground reflected irradiance on the front surface.
GroundReflectIrradiance = pvl_grounddiffuse(tilt, GHI, mean_albedo);

%%
% Calculate total irradiance on the front surface.
POAIrradiance = FrontSkyDiffuse + BeamIrradiance + GroundReflectIrradiance;

if make_figure
    figure
    hold all
    plot(dt, POAIrradiance)
    plot(dt, FrontSkyDiffuse)
    plot(dt, RearSkyDiffuse)
    legend('POA', 'Front sky diffuse', 'Rear sky diffuse')
    title(['Irradiance using ' SkyDiffuseModel ' sky diffuse model'])
    xlabel('Time of day')
    datetick('x', 'HH:MM', 'KeepTicks')
end

%% Set up control parameters for the irradiance simulation.

% Define edge of computation grid. |maxangle| is the angle between a vector
% looking down at the grid's center cell (horizontal on the ground) and
% a vector looking to the edge of the computational grid.
maxangle = 88; 

% Set the ratio between the viewing cell (e.g., a cell on the module or 
% a reference cell) and the ground grid cell dimension.  A value of 4
% means the ground grid is 1/4 the size of the viewing cell.
cell_ratio = 4;  

%% Simulate irradiance on rear-facing surface of each cell.
% Time series of rear-irradiance for each cell are 
% collected in structure |cellresult|. Rear irradiance is bundled with weather 
% data in structure |Result|.

Result = [];

%%
% Define grid of cells on the ground. The first element in the cells
% structure array is used to define the grid center on the ground.
% Normally set up so that cells(1) is the lower left cell in the
% recieving array.
gridcell = cells(1);

%%
% Scale the grid's cell.
gridcell.width = gridcell.width/cell_ratio;
gridcell.length = gridcell.length/cell_ratio;
gridcell.area = gridcell.area/(cell_ratio^2);

%%
% Create the grid.  The structure is named ggrid to avoid a namespace
% conflict with the builtin grid function. inputs vw and vl are
% characteristic unit vectors defining left-to-right and bottom-to-top,
% defined in conjuction with the objects
ggrid = create_grid(gridcell, maxangle, vw, vl);
ggrid.CC = calc_grid_cell_centers(ggrid);

%%
% Calculate VFs from each viewing cell to each cell in the ground grid.
h = waitbar(0, 'Calculating ground view factors...');
for i=1:length(cells)
    if mod(i, 10)==0
        waitbar(i/length(cells), h)
    end
    cells(i).VF = calc_approx_view_factors_grid(cells(i), ggrid);
end

% Make a figure showing VFs for one cell.
if make_figure
    figure
    view(3)
    hold all
    draw_grid_VFs(gca, ggrid, cells(1).VF);
    for i=1:length(objects)
        draw_object(gca, objects(i), true)
    end
    draw_cell(gca, cells(1), [1 0 0], false);
    axis equal
    grid on
    colorbar
    title(['View factors for ' cells(1).name])
end

close(h)


%%
% Calculate VFs from each ground grid cell to the sky, accounting for 
% blocking of portions of the sky by objects. A VF is calculated from 
% each ground grid cell to each object, the VFs are summed, and 
% 1-\Sum(VF) is the VF from the ground grid cell to the unblocked sky.
% This technique works only when objects don't overlap from the 
% perspective of a ground grid cell.

h = waitbar(0, 'Calculating sky view factors....');

tmp_skyVF = calc_sky_VF_vec(ggrid, objects(1));  
for i=2:length(objects)
    if mod(i, 3)==0
        waitbar(i/length(objects), h)
    end
    tmp_skyVF = tmp_skyVF + calc_sky_VF_vec(ggrid, objects(i));  % assumes that objects don't overlap
end
ggrid.skyVF = 1 - tmp_skyVF;
%
% To turn off sky blocking, set ggrid.skyVF = 1 for all grid cells.
% ggrid.skyVF = ones(size(cells(1).VF)); % turn off sky blocking

if make_figure
    figure
    view(3)
    hold all
    draw_grid_VFs(gca, ggrid, ggrid.skyVF);
    for i=1:length(objects)
        draw_object(gca, objects(i), true)
    end
    axis equal
    grid on
    colorbar
    title('Ground to sky view factors')
end

close(h)

%%
% Loop over times to calculate shadows from objects onto the ground.
% For each time, a filter |u| (array of 0s and 1s) is determined where
% |u(i,j) = 1| if ground grid cell |(i,j)| is shaded.

clear shadow_filt F;
% Compute array of vectors (x, y, z) from earth to the sun.
SunVec = [cosd(SunElevation).*sind(SunAzimuth) ...
          cosd(SunElevation).*cosd(SunAzimuth) ...
          sind(SunElevation)];

% This step takes some time.  Show a waitbar with calculation progress.
h = waitbar(0, 'Calculating ground shadows...');

for i=1:length(dt)
    % before_noon and after_noon are lists of object indices.  If i is
    % in list before_noon, then object i casts a shadow when the sun is
    % before solar noon.  This is how shadows from vertical objects 
    % are managed.
    if dv(i,4)<12
        objlist = before_noon;
    else
        objlist = after_noon;
    end

    % Update waitbar.
    if mod(i,10)==0
        waitbar(i/length(dt), h);
    end
    % Compute list of shadowed grid cells at each time step.
    % shadow_filt(i).u(i,j)=True means that ggrid cell
    % (i,j) is shaded at time dt(i).
%     shadow_filt(i).u = select_grid_points_inpoly(ggrid, objects(objlist), ...
%                                                  SunVec(i,:));
    shadow_filt(i).u = select_grid_points_InPolygon(ggrid, objects(objlist), ...
                                                 SunVec(i,:));

end

close(h)


%%
% Calculate rear-surface irradiance at each cell and each time.
% Strucure array |cellresult| contains rear-surface irradiance for each
% cell. 
% |cellresult| has the following fields:
% * |GRIrrad| : vector of ground reflected irradiance at each time.
% * |RearIrrad| : vector of total rear-surface irradiance at each time.
clearvars cellresult timeresult

h = waitbar(0, 'Calculating rear-surface irradiance by cell...');

% Set up data space
for j=1:length(cells)
    cellresult(j).GRIrrad = NaN(size(dt));
    cellresult(j).GRIrrad = cellresult(j).GRIrrad(:);
    cellresult(j).RearIrrad = cellresult(j).GRIrrad;
    cellresult(j).RearDirect = cellresult(j).GRIrrad;
    cellresult(j).cell_idx = j;
    cellresult(j).name = cells(j).name;
end

% Calculate ground reflected irradiance by cell
for i=1:length(dt)
    if mod(i, 10)==0
        waitbar(i/length(dt), h);
    end
    % Calculate irradiance on each ground grid cell
    timeresult(i).irrad = (1 - shadow_filt(i).u).*DNI(i)*cosd(90 - SunElevation(i)) ...
          +  DHI(i)*ggrid.skyVF;
    % Calculate ground-reflected (GRIrrad) irradiance on each cell
    for j=1:length(cells)
        cellresult(j).GRIrrad(i) = ggrid.cell_area/cells(j).area ... 
                           * mean_albedo*sum(sum(cells(j).VF.*timeresult(i).irrad));
    end
end


% Calculate total rear irradiance (sky diffuse + ground reflected + rear direct)
for j=1:length(cells)
    % Calculate direct (RearDirect) irradiance on each cell
    cosAOIrear = -cosd(AOI); % angle of incidence on rear surface
    cosAOIrear(cosAOIrear<0) = 0; % no direct if cos(AOI)<0
    cellresult(j).RearDirect = DNI.*cosAOIrear;
    cellresult(j).RearIrrad = RearSkyDiffuse + cellresult(j).GRIrrad + cellresult(j).RearDirect;
end

close(h)

%%
% Package |cellresult| with GHI, DNI and DHI in structure |Result| and 
% write to a file.
Result.cellresult = cellresult;
Result.dt = dt;
Result.hrofday = hrofday;
Result.GHI = GHI;
Result.DNI = DNI;
Result.DHI = DHI;

% Save |Result| to a file.
outfilen = ['Fixed_tilt_sim.mat'];
save(outfilen, 'Result')


%% Plot irradiance at each cell 

for k=1:size(which_modules, 1)

    row = which_modules(k, 1);
    pos = which_modules(k, 2);
    namestr = ['Row' num2str(row) ' Pos' num2str(pos)];
    
    % find elements in Result.cellresult which contain cells for PVID3179
    tmp = struct2table(Result.cellresult);
    u = strncmp(tmp.name, namestr, length(namestr));
    cellidx = tmp.cell_idx(u);
    
    figure
    hold all
    for kk=1:length(cellidx)
        plot(dt, Result.cellresult(cellidx(kk)).RearIrrad)
    end
    xlabel('Time of day')
    ylabel('W/m^2');
    datetick('x', 'HH:MM', 'KeepTicks')
    title({['Simulated irradiance at rear-facing cells for Row ' num2str(row) ', Pos ' num2str(pos)]})
end

