function grid = create_grid(cell, maxangle, vw, vl)

% CREATE_GRID returns a grid of square cells on the vw, vl plane.
%
% Syntax
%   [VF, A1] = create_grid(cell, vw, vl) 
% 
% Description
% 
%   create_grid defines a grid of square cells on the x-y plane which could
%   contribute to irradiance on the input cell's surface. The grid is
%   'centered' at the projection of the cell to the x,y plane.   The
%   returned grid is aligned with vw in its x-direction and vl in its 
%   y-direction. Grid cells have area equal to the cell's area.
%
% Inputs
%   cell - a structure defining the cell viewing the grid with the following fields:
%   cell.x, .y, .z - earth surface coordinates of the cell's center point
%   cell.normal - normal vector for cell pointing to the grid
%   cell.area = area of the cell
%   maxangle - maximum angle from cell center to edge of grid
%   vw, vl - unit vectors in the lateral (left-to-right) and vertical 
%      (bottom-to-top) directions of the cell 
%
% Output
%   grid - a structure array defining each grid cell with the following
%   fields:
%
%

% set limit for number to be different from zero
% required for condition checking due to small roundoff errors
eps = 1e-9;

% check that cell normal, vl and vw are orthogonal
if not(abs(dot(cell.normal, vl))<=eps && ...
       abs(dot(cell.normal, vw))<=eps && ...
       abs(dot(vl, vw))<=0)
    error('create_grid : cell.normal, vl and vw are not mutually orthogonal')
end

if cell.z<0
    error('create_grid : cell.z is negative')
end


cell_center = [cell.x cell.y cell.z];

% get unit normal to the cell in the direction of the earth surface
n1 = cell.normal;
if n1(3)>0
    n1 = -n1;
end

% define origin of the grid.

if n1(1)==0 && n1(2)==0
    % cell is parallel to the earth, project center to surface along n1
    origin = [cell_center(1:2) 0];
    cell_orientation = 'horizontal';
    % unit vectors for the grid
    gw = vw(1:2)/norm(vw(1:2));
    gl = vl(1:2)/norm(vl(1:2));
elseif n1(3)==0
    % cell is vertical, project center to surface along -vl
%    error('create_grid: vertical cells not implemented')
    origin = cell_center; origin(3) = 0;
    cell_orientation = 'vertical';
    % unit vectors for the grid
    gw = vw(1:2)/norm(vw(1:2));
    gl = cell.normal(1:2)/norm(cell.normal(1:2));
else
    % assume cell's 'x'-edge (width) is parallel to the earth
    if not(vw(3)==0)
        error('create_grid : vw(3) must be 0, tilted cell must have width parallel to earth')
    end
    d = dot(n1, [0 0 -1]);
    % set grid origin at projection of center to x,y along normal
    origin = cell_center + cell.z/d*n1;
    cell_orientation = 'tilted';
    % unit vectors for the grid
    gw = vw(1:2)/norm(vw(1:2));
    gl = vl(1:2)/norm(vl(1:2));
end

if not(abs(origin(3))<eps)
    error('create_grid : error calculating origin, z coordinate not 0')
end

% origin for the grid, not the center of the cell at the origin
origin = origin(1:2);

grid.cell_area = cell.area;
dw = sqrt(cell.area);
dl = dw;

% determine extent of grid
max_theta = maxangle;

% number of cells to reach edge of cone with half angle max_theta
W = ceil(cell.z/(dw*tand(90 - max_theta)));


% compute cell centers in earth reference coordinates
% cx = NaN(2*W+1, W+1);
% cy = cx;
% for i=-W:1:W
%     for j=0:1:W;
%         cc = cell00 + i*dw*gw + j*dl*gl;
%         iidx = W+i+1;
%         jidx = j+1;
%         cx(iidx,jidx) = cc(1);
%         cy(iidx,jidx) = cc(2);
%     end
% end

% grid is symmetric around center point
cx = dw*(-W:1:W);
cy = dl*(-W:1:W);

% place 0,0 cell at origin + dl/2 * gl (so that the bottom edge of the cell
% lies along the lateral axis)

offset = dl/2*gl;
cell00 = origin + offset;  % center of cell at origin

grid.origin = origin;
grid.origin_x_idx = W;
grid.origin_y_idx = W;
grid.cell00 = cell00; % center of cell with edge at grid origin
grid.cell_orientation = cell_orientation;
grid.centers_x = cx;  % distance from origin along grid vector gw to cell center
grid.centers_y = cy;  % distance from origin along grid vector gl to cell center
grid.vw = gw; % unit vector in the grid's width direction
grid.vl = gl; % unit vector in the grid's length direction
grid.dw = dw; % cell dimension in the width direction
grid.dl = dl; % cell dimension in the length direction



