function Omega = calc_solid_angle_vec(viewpoint, obj)

% calculates the solid angle Omega subtended by the rectangle in obj for
% a sphere centered at viewpoint.

% input viewpoint is a Nx3 array of points in 3D Cartesian coordinates
% input obj must be a struct with the following fields:
%  - obj.LL, .LR, .UR, .UL : defines the corners of the rectangle in 3D
%  Cartesian coordinates
%  - obj.normal : normal vector to the plane defined by obj

% Reference: R. J. Mathar, Solid Angle of a Rectangular Plate, May 18, 2015
% Downloaded May 5 2017 from http://www.mpia.de/~mathar/public/mathar20051002.pdf

% calculate rectangle dimensions
a = norm(obj.LR - obj.LL);
b = norm(obj.UR - obj.UL);
 
% number of viewpoints to consider
N = size(viewpoint, 1);

% vectorize object
vLL = repmat(obj.LL,N,1);

% determine normal pointing in the direction of viewpoint
n1 = repmat(obj.normal,N,1);
u = dot(n1, vLL - viewpoint)>0;  % n1 points away from viewpoint
n1(u,:) = -n1(u,:);
 
% calculate point V in obj plane nearest to the viewpoint
d = dot(n1, (viewpoint - vLL), 2);
V = viewpoint - repmat(d,1,3).*n1;

% determine case for calculation of solid angle

% plane coordinate system with origin at obj.LL
vw = (obj.LR - obj.LL)/a;
vl = (obj.UL - obj.LL)/b;

% locate V within plane coordinate system
projV_w = dot(V - vLL, repmat(vw,N,1), 2);
projV_l = dot(V - vLL, repmat(vl,N,1), 2);


inside_w_edge = (projV_w > 0) & (projV_w < a);
inside_l_edge = (projV_l > 0) & (projV_l < b);
A = min(abs(projV_w), abs(projV_w - a));
B = min(abs(projV_l), abs(projV_l - b));

Omega = NaN(N,1);

u = ~inside_w_edge & ~inside_l_edge;
% Case 1: rectangle is entirely within one quadrant relative to V
Omega(u) = 0.25 * (omega(2*(A(u) + a), 2*(B(u) + b), d(u)) ...
              - omega(2*A(u), 2*(B(u) + b), d(u)) ...
              - omega(2*(A(u) + a), 2*B(u), d(u)) ...
              + omega(2*A(u), 2*B(u), d(u)) );
u = inside_w_edge & ~inside_l_edge;
% Case 2: rectangle intersects w axis and spans 2 quadrants relative to V
Omega(u) = 0.25 * (omega(2*(A(u) - a), 2*(b + B(u)), d(u)) ...
              - omega(2*(a - A(u)), 2*B(u), d(u)) ...
              + omega(2*A(u), 2*(b + B(u)), d(u)) ...
              - omega(2*A(u), 2*B(u), d(u)) );
u = ~inside_w_edge & inside_l_edge;
% Case 2: rectangle intersects l axis and spans 2 quadrants relative to V
Omega(u) = 0.25 * (omega(2*(a + A(u)), 2*(b - B(u)), d(u)) ...
              - omega(2*A(u), 2*(b - B(u)), d(u)) ...
              + omega(2*(a + A(u)), 2*B(u), d(u)) ...
              - omega(2*A(u), 2*B(u), d(u)) );
u = inside_w_edge & inside_l_edge;
% Case 3: rectangle contains V
Omega(u) = 0.25 * (omega(2*(a - A(u)), 2*(b - B(u)), d(u)) ...
              + omega(2*A(u), 2*(b - B(u)), d(u)) ...
              + omega(2*(a - A(u)), 2*B(u), d(u)) ...
              + omega(2*A(u), 2*B(u), d(u)) );


function y = omega(a, b, d)

% calculates solid angle given cone parameters

alpha = a./(2*d);
beta = b./(2*d);

y = 4 * acos(sqrt((1 + alpha.^2 + beta.^2)./((1 + alpha.^2).*(1 + beta.^2))));
